/*******************************************************************************
 mod_h264_streaming - A lighttpd plugin for pseudo-streaming Quicktime/MPEG4 files.

 Copyright (C) 2007-2021 CodeShop B.V.

 For licensing see the LICENSE file
******************************************************************************/

// Example configuration

/*
# Add the module to your server list in lighttpd.conf
server.modules = (
  "mod_expire",
  "mod_secdownload",
  "mod_h264_streaming",
  etc...
}

# Files ending in .mp4 and .f4v are served by the module
h264-streaming.extensions = ( ".mp4", ".f4v" )
# The number of seconds after which the bandwidth is shaped (defaults to 0=disable)
h264-streaming.buffer-seconds = 10

# Add Expires/Cache-Control header
$HTTP["url"] =~ "\.(mp4|f4v)$" {
  expire.url = ( "" => "access 8 hours" )
}

*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "base.h"
#include "log.h"
#include "buffer.h"
#include "response.h"
#if LIGHTTPD_VERSION_ID < ((1 << 16) + (5 << 8))
#include "http_chunk.h"
#else
#include <unistd.h>
#endif
#include "stat_cache.h"
#include "etag.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <mp4_process.h>
#include <mp4_rewrite.h>
#include <mp4_options.h>
#include <moov.h>
#include <output_bucket.h>
#define X_MOD_STREAMING_KEY X_MOD_SMOOTH_STREAMING_KEY
#define X_MOD_STREAMING_VERSION X_MOD_SMOOTH_STREAMING_VERSION
#ifdef HAVE_FMP4_LIVE
#include <mp4_pubpoint.h>
#include <http_util.h>
#include "sys-mmap.h"
#endif

#include <assert.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>

/* plugin config for all request/connections */

typedef struct {
  array *extensions;
  unsigned short buffer_seconds;
  buffer* license;
} plugin_config;

typedef struct {
  PLUGIN_DATA;

  plugin_config **config_storage;

  plugin_config conf;

  mp4_global_context_t* gctx;
} plugin_data;

typedef struct {
  struct mp4_process_context_t context_;
  int play_position; /* (in seconds) */
  unsigned short buffer_seconds;

#if defined(HAVE_FMP4_LIVE)
  post_handler_t* post_handler;
#endif
} handler_ctx;

static handler_ctx * handler_ctx_init(unsigned short buffer_seconds,
                                      mp4_global_context_t* gctx)
{
  handler_ctx * hctx;

  hctx = calloc(1, sizeof(*hctx));

  mp4_process_context_init(&hctx->context_, gctx);
  hctx->play_position = 0;
  hctx->buffer_seconds = buffer_seconds;
#if defined(HAVE_FMP4_LIVE)
  hctx->post_handler = 0;
#endif

  return hctx;
}

static void handler_ctx_free(handler_ctx *hctx)
{
  mp4_process_context_exit(&hctx->context_);

#if defined(HAVE_FMP4_LIVE)
  if(hctx->post_handler)
  {
    post_handler_exit(hctx->post_handler);
  }
#endif

  free(hctx);
}

void init_policy(plugin_data* p, server* srv)
{
  char const* src = "lighttpd mod_smooth_streaming";
  char const* version = X_MOD_SMOOTH_STREAMING_VERSION;

  char const* license = 0;
  if (p->conf.license->ptr) {
    license = p->conf.license->ptr;
  }

  char const* policy_check_result =
    libfmp4_load_license(p->gctx, src, version, license);

  if(policy_check_result != 0) {
    log_error_write(srv, __FILE__, __LINE__, "s", policy_check_result);
  } else if(license) {
    log_error_write(srv, __FILE__, __LINE__, "ss", "License key found: ", license);
  }
}

/* init the plugin data */
INIT_FUNC(mod_h264_streaming_init)
{
  plugin_data *p;

  p = calloc(1, sizeof(*p));

  p->gctx = libfmp4_global_init();

  return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_h264_streaming_free) {
  plugin_data *p = p_d;

  UNUSED(srv);

  libfmp4_global_exit(p->gctx);

  if (!p) return HANDLER_GO_ON;

  if (p->config_storage) {
    size_t i;

    for (i = 0; i < srv->config_context->used; i++) {
      plugin_config *s = p->config_storage[i];

      if (!s) continue;

      array_free(s->extensions);
      buffer_free(s->license);

      free(s);
    }
    free(p->config_storage);
  }

  free(p);

  return HANDLER_GO_ON;
}

/* handle plugin config and check values */

#define CONFIG_SMOOTHSTREAMING_EXTENSIONS   "smooth-streaming.extensions"
#define CONFIG_SMOOTHSTREAMING_USP_LICENSE_KEY   "smooth-streaming.usp_license_key"

SETDEFAULTS_FUNC(mod_h264_streaming_set_defaults) {
  plugin_data *p = p_d;
  size_t i = 0;

  config_values_t cv[] = {
    { CONFIG_SMOOTHSTREAMING_USP_LICENSE_KEY, NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_SERVER },
    { CONFIG_SMOOTHSTREAMING_EXTENSIONS,      NULL, T_CONFIG_ARRAY, T_CONFIG_SCOPE_CONNECTION }, /* 0 */
    { NULL,                                   NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
  };

  if (!p) return HANDLER_ERROR;

  p->config_storage = calloc(1, srv->config_context->used * sizeof(specific_config *));

  for (i = 0; i < srv->config_context->used; i++) {
    plugin_config *s;

    s = calloc(1, sizeof(plugin_config));
    s->extensions     = array_init();
    s->buffer_seconds = 0;
    s->license = buffer_init();

    cv[0].destination = s->license;
    cv[1].destination = s->extensions;

    p->config_storage[i] = s;

    if (0 != config_insert_values_global(srv, ((data_config *)srv->config_context->data[i])->value, cv)) {
      return HANDLER_ERROR;
    }
  }

  p->conf.license = p->config_storage[0]->license;

  init_policy(p, srv);

  return HANDLER_GO_ON;
}

#if LIGHTTPD_VERSION_ID < ((1 << 16) + (5 << 8))
#define PATCH_OPTION(x) \
p->conf.x = s->x;
#endif

static int mod_h264_streaming_patch_connection(server *srv, connection *con, plugin_data *p)
{
  size_t i, j;
  plugin_config *s = p->config_storage[0];

  PATCH_OPTION(extensions);
  PATCH_OPTION(buffer_seconds);

  /* skip the first, the global context */
  for (i = 1; i < srv->config_context->used; i++) {
    data_config *dc = (data_config *)srv->config_context->data[i];
    s = p->config_storage[i];

    /* condition didn't match */
    if (!config_check_cond(srv, con, dc)) continue;

    /* merge config */
    for (j = 0; j < dc->value->used; j++) {
      data_unset *du = dc->value->data[j];

      if (buffer_is_equal_string(du->key, CONST_STR_LEN(CONFIG_SMOOTHSTREAMING_EXTENSIONS))) {
        PATCH_OPTION(extensions);
      }
    }
  }

  return 0;
}
#undef PATCH_OPTION

#if defined(HAVE_FMP4_LIVE)
URIHANDLER_FUNC(mod_h264_streaming_uri_handler)
{
  plugin_data *p = p_d;

  if (con->uri.path->used == 0)
    return HANDLER_GO_ON;

  mod_h264_streaming_patch_connection(srv, con, p);

  // are we setup for this virtual server?
  if(!p->conf.extensions->used)
  {
    return HANDLER_GO_ON;
  }

  if (con->conf.log_request_handling)
  {
    log_error_write(srv, __FILE__, __LINE__, "s",
                    "-- mod_h264_streaming_uri_handler called");
  }

  {
    char path[512];
    char args[512];
    unsigned int rewrite_ism = 1;
    unsigned int rewrite_f4f = 1;

    char const* uri_first = con->uri.path->ptr;
    char const* uri_last = uri_first +
      (con->uri.path->used == 0 ? 0 : (con->uri.path->used - 1));
    char const* query_first = con->uri.query->ptr;
    char const* query_last = query_first +
      (con->uri.query->used == 0 ? 0 : (con->uri.query->used - 1));

    if(!mp4_rewrite_url(uri_first, uri_last,
                        query_first, query_last,
                        path, sizeof(path), args, sizeof(args),
                        rewrite_ism, rewrite_f4f))
    {
      return HANDLER_GO_ON;
    }

    if (con->conf.log_request_handling)
    {
      log_error_write(srv, __FILE__, __LINE__, "sbbss",
                      "mp4_rewrite:",
                      con->uri.path, con->uri.query,
                      path, args);
    }

    buffer_copy_string(con->uri.path, path);
    buffer_copy_string(con->uri.query, args);

    return HANDLER_GO_ON;
  }
}
#endif

#if defined(HAVE_FMP4_LIVE)
static int push_input_stream(server* srv, connection* con,
                             post_handler_t* post_handler)
{
  int result = 200;

#if LIGHTTPD_VERSION_ID < ((1 << 16) + (5 << 8))
  chunkqueue *cq = con->request_content_queue;
#else
  chunkqueue *cq = con->recv;
#endif

  {
    chunk *c;

    /* there is content to send */
    for (c = cq->first; c; c = cq->first) {
      char const* first = NULL;
      char const* last = NULL;

      switch(c->type) {
      case FILE_CHUNK:

        if (c->file.mmap.start == MAP_FAILED) {
          if (-1 == c->file.fd &&  /* open the file if not already open */
              -1 == (c->file.fd = open(c->file.name->ptr, O_RDONLY))) {
            log_error_write(srv, __FILE__, __LINE__, "ss", "open failed: ", strerror(errno));
            return 500;
          }

          c->file.mmap.length = c->file.length;

          if (MAP_FAILED == (c->file.mmap.start = mmap(0,  c->file.mmap.length, PROT_READ, MAP_SHARED, c->file.fd, 0))) {
            log_error_write(srv, __FILE__, __LINE__, "ssbd", "mmap failed: ",
                strerror(errno), c->file.name,  c->file.fd);
            return 500;
          }

          close(c->file.fd);
          c->file.fd = -1;

          /* chunk_reset() or chunk_free() will cleanup for us */
        }

        first = c->file.mmap.start + c->offset;
        last = c->file.mmap.start + c->file.length;
        break;
      case MEM_CHUNK:
        first = c->mem->ptr + c->offset;
        last = c->mem->ptr + c->mem->used - 1;
        break;
      case UNUSED_CHUNK:
        break;
      }

      if(first && last)
      {
        char result_text[256];

        result = post_handler_insert(post_handler,
          (unsigned char const*)first, (unsigned char const*)last,
          result_text, sizeof(result_text) / sizeof(result_text[0]));

        if(result != 200)
        {
          log_error_write(srv, __FILE__, __LINE__, "ss",
                          "-- pubpoint_push_input_stream returned",
                          result_text);
          break;
        }

        c->offset += (last - first);
        cq->bytes_out += (last - first);
      }

      chunkqueue_remove_finished_chunks(cq);
    }
  }

  return result;
}
#endif

#if defined(HAVE_FMP4_LIVE)
URIHANDLER_FUNC(mod_h264_streaming_send_request_content_handler)
{
#if LIGHTTPD_VERSION_ID < ((1 << 16) + (5 << 8))
  chunkqueue *in = con->request_content_queue;
#else
  chunkqueue *in = con->recv;
#endif
  plugin_data* p = p_d;
  handler_ctx *hctx = con->plugin_ctx[p->id];

  if(hctx == NULL || hctx->post_handler == NULL)
    return HANDLER_GO_ON;

#if LIGHTTPD_VERSION_ID >= ((1 << 16) + (5 << 8))
  /* there is nothing that we have to send out anymore */
  if (in->bytes_in == in->bytes_out &&
      in->is_closed) return HANDLER_GO_ON;
#endif

  if (con->conf.log_request_handling)
  {
    log_error_write(srv, __FILE__, __LINE__, "sososo",
        "-- push_input_stream: ", in->bytes_out, "-", in->bytes_in,
        "/", con->request.content_length);
  }

//  if(!mp4_ends_with(filename, ".ism"))
//  {
//    con->http_status = 403; // HTTP_FORBIDDEN;
//    return HANDLER_FINISHED;
//  }

  {
    int result = push_input_stream(srv, con, hctx->post_handler);

    con->http_status = result;
  }

  return HANDLER_GO_ON;
}
#endif

static int our_extension(connection* con, plugin_data* p)
{
  int s_len;
  size_t k;

  s_len = con->physical.path->used - 1;

  for (k = 0; k < p->conf.extensions->used; k++)
  {
    data_string *ds = (data_string *)p->conf.extensions->data[k];
    int ct_len = ds->value->used - 1;

    if (ct_len > s_len) continue;
    if (ds->value->used == 0) continue;

    if (0 == strncmp(con->physical.path->ptr + s_len - ct_len, ds->value->ptr, ct_len))
      return 1;
  }

  return 0;
}

URIHANDLER_FUNC(mod_h264_streaming_path_handler)
{
  plugin_data *p = p_d;
  handler_ctx *hctx;
  int rc;
//int ours;
  stat_cache_entry *sce = NULL;
  mp4_split_options_t* options;

#if LIGHTTPD_VERSION_ID < ((1 << 16) + (5 << 8))
  chunkqueue *cq = con->write_queue;
#else
  chunkqueue *cq = con->send;
#endif

#if LIGHTTPD_VERSION_ID >= (1 << 16 | 4 << 8 | 28) && LIGHTTPD_VERSION_ID < (1 << 16 | 5 << 8)
  if (con->mode != DIRECT)
    return HANDLER_GO_ON;
#endif

  if (buffer_is_empty(con->physical.path))
  {
    return HANDLER_GO_ON;
  }

  mod_h264_streaming_patch_connection(srv, con, p);

  if (con->conf.log_request_handling)
  {
    log_error_write(srv, __FILE__, __LINE__, "s",
                    "-- mod_h264_streaming_path_handler called");
  }

  if(!our_extension(con, p))
  {
    if (con->conf.log_request_handling)
    {
      log_error_write(srv, __FILE__, __LINE__, "sdsb", "none of the", p->conf.extensions->used,  "extensions matched",
                      con->physical.path);
    }
    return HANDLER_GO_ON;
  }

  rc = stat_cache_get_entry(srv, con, con->physical.path, &sce);

  if (rc == HANDLER_GO_ON &&
      (con->request.http_method == HTTP_METHOD_GET ||
       con->request.http_method == HTTP_METHOD_HEAD))
  {
    if(S_ISDIR(sce->st.st_mode))
    {
      char const* path_first = con->physical.path->ptr;
      char const* path_last = path_first +
        (con->physical.path->used == 0 ? 0 : (con->physical.path->used - 1));
      char path[512];
      mp4_rewrite_dir(path_first, path_last, path, sizeof(path));

      if (con->conf.log_request_handling)
      {
        log_error_write(srv, __FILE__, __LINE__, "sbbs",
                        "mp4_rewrite:",
                        con->physical.path, con->uri.query,
                        path);
      }

      buffer_copy_string(con->physical.path, path);
      rc = stat_cache_get_entry(srv, con, con->physical.path, &sce);
    }
  }

  // if file does not exist, let core handle it
  if (rc != HANDLER_GO_ON &&
      (con->request.http_method == HTTP_METHOD_GET || con->request.http_method == HTTP_METHOD_HEAD))
  {
    return HANDLER_GO_ON;
  }

  // Module version info
  response_header_overwrite(srv, con,
    CONST_STR_LEN("X-USP"),
    fmp4_version_string(), strlen(fmp4_version_string()));

  if (con->plugin_ctx[p->id]) {
    hctx = con->plugin_ctx[p->id];
  } else {
    hctx = handler_ctx_init(p->conf.buffer_seconds, p->gctx);
    con->plugin_ctx[p->id] = hctx;
  }

  options = hctx->context_.options_;

  if(con->uri.query->used &&
     !mp4_split_options_set(options, con->uri.query->ptr,
                            con->uri.query->used - 1))
  {
    con->http_status = 400;
    return HANDLER_FINISHED;
  }

  switch(con->request.http_method)
  {
#if defined(HAVE_FMP4_LIVE)
  case HTTP_METHOD_POST:
  {
    hctx->context_.filename_ = con->physical.path->ptr;

    hctx->post_handler = pubpoint_create_post_handler(&hctx->context_);

    if(hctx->post_handler == NULL)
    {
      log_error_write(srv, __FILE__, __LINE__, "ss",
                      "-- create_post_handler returned",
                      hctx->context_.result_text_);
      con->http_status = 403; // HTTP_FORBIDDEN;
      return HANDLER_FINISHED;
    }

#if LIGHTTPD_VERSION_ID >= ((1 << 16) + (5 << 8))
    cq->is_closed = 1;
#else
    mod_h264_streaming_send_request_content_handler(srv, con, p_d);
    con->file_finished = 1;
#endif
    return HANDLER_FINISHED;
  }
#endif
  case HTTP_METHOD_GET:
  case HTTP_METHOD_HEAD:
    // If we are requesting the server manifest file, then it may be an encoding tool
    // that is using the GET request to get the authorization setup for the POST.
    if(mp4_ends_with(con->physical.path->ptr, ".isml"))
    {
      char const* file = mp4_split_options_get_file(options);
      if(mp4_starts_with(file, "Streams("))
      {
        con->http_status = 200;
        return HANDLER_FINISHED;
      }

//      // we are either serving dynamically generated files, or fragments
//      if(!file[0])
//      {
//        con->http_status = 403; // HTTP_FORBIDDEN;
//        return HANDLER_FINISHED;
//      }
    }
    break;
  default:
    con->http_status = 405; // HTTP_METHOD_NOT_ALLOWED;
    return HANDLER_FINISHED;
  }

  {
    buffer *mtime = strftime_cache_get(srv, sce->st.st_mtime);
//    stat_cache_entry *sce = NULL;

#if defined(BUILDING_H264_STREAMING)
    // Range requests are currently not supported, so let mod_staticfile
    // handle it. Obviously the 'start' parameter doesn't work with
    // mod_staticfile and so the movie always plays from the beginning,
    // but at least it makes streaming and seeking in VLC work.
#if LIGHTTPD_VERSION_ID < ((1 << 16) + (5 << 8))
    if (con->conf.range_requests &&
       NULL != array_get_element(con->request.headers, "Range")) {
        return HANDLER_GO_ON;
    }
#else
    if (con->conf.range_requests &&
       NULL != array_get_element(con->request.headers, CONST_STR_LEN("Range"))) {
        return HANDLER_GO_ON;
    }
#endif
#else
    // If the server ignores a byte-range-spec because it is syntactically
    // invalid, the server SHOULD treat the request as if the invalid Range
    // header field did not exist. (Normally, this means return a 200 response
    // containing the full entity).
#endif

//    response_header_overwrite(srv, con,
//      CONST_STR_LEN(X_MOD_STREAMING_KEY),
//      CONST_STR_LEN(X_MOD_STREAMING_VERSION));


    /* we are safe now, let's build a h264 header */
    {
      mp4_process_context_t* context = &hctx->context_;
      int http_status;

      context->filename_ = con->physical.path->ptr;
//    context->filesize_ = sce->st.st_size;

      http_status = mp4_process(context);

      headers_t* headers = context->headers_;
      buckets_t* buckets = context->buckets_;

      if(headers->x_usp_header1[0])
      {
        response_header_overwrite(srv, con, CONST_STR_LEN("X-USP-Info1"),
                                  headers->x_usp_header1,
                                  strlen(headers->x_usp_header1));
      }
      if(headers->x_usp_header2[0])
      {
        response_header_overwrite(srv, con, CONST_STR_LEN("X-USP-Info2"),
                                  headers->x_usp_header2,
                                  strlen(headers->x_usp_header2));
      }

      if(http_status != 200)
      {
        log_error_write(srv, __FILE__, __LINE__, "sss",
                        "-- mp4_process returned",
                        fmp4_result_to_string(context->result_),
                        context->result_text_);
        con->http_status = http_status;
        return HANDLER_FINISHED;
      }

      // Add last-modified header
      if(headers->updated_at_)
      {
        mtime = strftime_cache_get(srv, headers->updated_at_ / 1000000);
        response_header_overwrite(srv, con, CONST_STR_LEN("Last-Modified"),
                                  CONST_BUF_LEN(mtime));
      }

      if(headers->cache_control_)
      {
        response_header_overwrite(srv, con, CONST_STR_LEN("Cache-Control"),
                                  headers->cache_control_,
                                  strlen(headers->cache_control_));
      }
      else if(headers->expires_at_)
      {
        time_t expires_time = headers->expires_at_ / 1000000;
        time_t request_time = srv->cur_ts;

        // use expiry from backend only if it's at least a second ahead of request
        // otherwise, we skip insertion of Expires/Cache-Control and leave this up
        // to the customer
        if(expires_time < request_time + 1)
        {
          expires_time = request_time + 1;
        }

        unsigned int max_age = expires_time - request_time;
        char strbuf[256];
        int len = sprintf(strbuf, "max-age=%u", max_age);
        response_header_overwrite(srv, con, CONST_STR_LEN("Cache-Control"),
                                  strbuf, len);

        buffer* etime = strftime_cache_get(srv, expires_time);
        response_header_overwrite(srv, con, CONST_STR_LEN("Expires"),
                                  CONST_BUF_LEN(etime));
      }
      response_header_overwrite(srv, con, CONST_STR_LEN("Content-Type"),
                                headers->content_type_,
                                strlen(headers->content_type_));

      // Add the link header for start/next
      if(headers->link_[0])
      {
        response_header_overwrite(srv, con, CONST_STR_LEN("Link"),
                                  headers->link_,
                                  strlen(headers->link_));
      }

      // Set the content length
      {
        uint64_t content_length = buckets_size(buckets);

        if(content_length == UINT64_MAX)
        {
          con->http_status = 501; /* Not Implemented */
          return HANDLER_FINISHED;
        }
      }

      {
        bucket_t* head = buckets->bucket_;
        bucket_t* bucket = bucket_next(head);
        if(head != bucket)
        {
          buffer* filename = buffer_init_buffer(con->physical.path);
          for(; bucket != head; bucket = bucket_next(bucket))
          {
            if(is_bucket_type_file(bucket))
            {
              char const* fname;
              uint64_t offset;
              uint64_t size = 0;
              uint32_t max_size = 16 * 1024 * 1024;
              bucket_file_read(bucket, &fname, &offset, &size, max_size);

              buffer_copy_string(filename, fname);

              // convert from a escaped file:// URL to a local path
              filename->used = file_url_to_path(filename->ptr) - filename->ptr + 1;

              chunkqueue_append_file(cq, filename, offset, size);
            }
            else
            {
              uint8_t const* src = 0;
              size_t size = 0;
              bucket_read(bucket, &src, &size);
              buffer* b = chunkqueue_get_append_buffer(cq);
              buffer_append_memory(b, (char const *) src, size);
              b->used++; /* add virtual \0 */
            }
          }
          buffer_free(filename);
        }
      }

      // ETag header
      {
        if(headers->etag_[0])
        {
          buffer_copy_memory(con->physical.etag, headers->etag_,
                             strlen(headers->etag_) + 1);
        }
        else
        {
          etag_mutate(con->physical.etag, sce->etag);
        }
        response_header_overwrite(srv, con, CONST_STR_LEN("ETag"),
                                  CONST_BUF_LEN(con->physical.etag));
      }
    }

    if (HANDLER_FINISHED == http_response_handle_cachable(srv, con, mtime))
    {
      return HANDLER_FINISHED;
    }

#if LIGHTTPD_VERSION_ID >= ((1 << 16) + (5 << 8))
    cq->is_closed = 1;
#else
    con->file_finished = 1;
#endif

    if(hctx->buffer_seconds)
    {
      uint32_t seconds = fmp4_get_throttle_seconds(options);
      if(hctx->buffer_seconds < seconds)
      {
        uint64_t const* offsets = fmp4_get_throttle_offsets(options);
        con->conf.kbytes_per_second = offsets[hctx->buffer_seconds] / 1024;
      }
    }

    return HANDLER_FINISHED;
  }
}

TRIGGER_FUNC(mod_h264_streaming_trigger)
{
  plugin_data *p = p_d;
  size_t i;

  /* check all connections */
  for (i = 0; i < srv->conns->used; i++) {
    connection *con = srv->conns->ptr[i];
    handler_ctx *hctx = con->plugin_ctx[p->id];
    int t_diff = srv->cur_ts - con->connection_start;

    if(hctx == NULL)
      continue;

    // check if bandwidth shaping is enabled
    if(hctx->buffer_seconds)
    {
      mp4_split_options_t* options = hctx->context_.options_;
      int32_t seconds = fmp4_get_throttle_seconds(options);

      hctx->play_position = srv->cur_ts - con->connection_start;

      // when we are near the end you're no longer throttled
      if((t_diff + hctx->buffer_seconds) >= seconds)
      {
        con->conf.kbytes_per_second = 0;
        continue;
      }
      else
      {
        uint64_t const* offsets = fmp4_get_throttle_offsets(options);
        int64_t moov_offset = offsets[t_diff + hctx->buffer_seconds];

        if(con->bytes_written < moov_offset)
        {
          con->conf.kbytes_per_second = 1 + (moov_offset - con->bytes_written) / 1024;
        }
        else
        {
        // throttle connection to a low speed (set to the size of the TCP send
        // buffer) as the buffer is full
        con->conf.kbytes_per_second = 32;
        }

#if 0
        if (con->conf.log_request_handling) {
          TRACE("[%zu] con[%zu].conf.kbytes_per_second=%d limit=%d (%"PRId64"KB/%"PRId64"KB) this second=%"PRId64" play_pos=%d",
                p->id,
                i,
                con->conf.kbytes_per_second,
                con->traffic_limit_reached,
                (con->bytes_written >> 10),
                (moov_offset >> 10),
                con->bytes_written_cur_second,
                hctx->play_position);
        }
#endif
      }
    }
  }

  return HANDLER_GO_ON;
}

static handler_t mod_h264_streaming_cleanup(server *srv, connection *con, void *p_d)
{
  plugin_data *p = p_d;
  handler_ctx *hctx = con->plugin_ctx[p->id];

  UNUSED(srv);

  if(hctx == NULL)
    return HANDLER_GO_ON;

  handler_ctx_free(hctx);
  con->plugin_ctx[p->id] = NULL;

  return HANDLER_GO_ON;
}

/* this function is called at dlopen() time and inits the callbacks */

int mod_smooth_streaming_plugin_init(plugin *p)
{
  p->version     = LIGHTTPD_VERSION_ID;
  p->name        = buffer_init_string("unified_streaming");

  p->init        = mod_h264_streaming_init;
  p->set_defaults  = mod_h264_streaming_set_defaults;
#if defined(HAVE_FMP4_LIVE)
  p->handle_uri_clean = mod_h264_streaming_uri_handler;
#if LIGHTTPD_VERSION_ID >= ((1 << 16) + (5 << 8))
  p->handle_send_request_content = mod_h264_streaming_send_request_content_handler;
#endif
#endif
  p->handle_physical = mod_h264_streaming_path_handler;
  p->cleanup     = mod_h264_streaming_free;

  p->handle_trigger = mod_h264_streaming_trigger;

  p->connection_reset = mod_h264_streaming_cleanup;
  p->handle_connection_close = mod_h264_streaming_cleanup;

  p->data        = NULL;

  return 0;
}

