/*******************************************************************************
 ngx_http_h264_streaming_module.c - An Nginx module for streaming Quicktime/MPEG4 files.

 Copyright (C) 2008-2022 CodeShop B.V.

 For licensing see the LICENSE file
******************************************************************************/

// TODO
//#if !defined(HAVE_FMP4_LIVE) && !defined(FMP4_NO_INGEST_HTTP)
//#define FMP4_NO_INGEST_HTTP
//#endif

#include <nginx.h>
#include <ngx_config.h>
#include <ngx_core.h>
#include <ngx_http.h>
#include <fmp4/mp4_process.h>
#include <fmp4/mp4_rewrite.h>
#include <fmp4/mp4_options.h>
#include <fmp4/moov.h>
#include <fmp4/output_bucket.h>
#include <fmp4/mp4_handler_io.h>
#include <fmp4/post_handler.h>

#if 0
/* Mod-H264-Streaming configuration

server {
  listen 82;
  server_name  localhost;

  location ~ \.mp4$ {
    root /var/www;
    mp4;
  }
}

*/

/* Mod-Smooth-Streaming configuration

server {
  listen 82;
  server_name localhost;

  location ~ \.ism$ {
    root /var/www;
    usp_handle_ism;
  }
}
*/
#endif

typedef struct
{
  ngx_flag_t enable;

  // ism_proxy_pass
  ngx_str_t url;

  ngx_flag_t usp_prefer_static;
  ngx_flag_t usp_iss_pass_through;
  ngx_flag_t usp_dynamic_time_shift_buffer_depth;
  ngx_flag_t usp_skip_rewrite;

  ngx_flag_t usp_handle_f4f;
  ngx_flag_t usp_handle_api;
  ngx_flag_t usp_disable_mmap;

  ngx_str_t s3_secret_key;
  ngx_str_t s3_access_key;
  ngx_str_t s3_region;

} ngx_http_streaming_loc_conf_t;

typedef struct
{
  post_handler_t* handler;

} ngx_http_streaming_ctx_t;

typedef struct
{
  mp4_global_context_t* gctx;
  ngx_str_t license;

} ngx_http_streaming_main_conf_t;

static char* ngx_streaming(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static ngx_int_t ngx_streaming_handler_get(ngx_http_request_t *r,
                                           mp4_process_context_t* context);
#if defined(BUILDING_SMOOTH_STREAMING)
static ngx_int_t ngx_http_streaming_init(ngx_conf_t* cf);
static ngx_int_t ngx_streaming_handler_post(ngx_http_request_t *r,
                                            mp4_process_context_t* context);
static char* ngx_ism_proxy_pass(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_prefer_static(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_iss_pass_through(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_dynamic_time_shift_buffer_depth(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_skip_rewrite(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_handle_f4f(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_handle_api(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_disable_mmap(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_s3_secret_key(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_s3_access_key(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static char* ngx_usp_s3_region(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);

#endif
static char* ngx_usp_license_key(ngx_conf_t* cf, ngx_command_t* cmd, void* conf);
static fmp4_http_method_t ngx_get_method(ngx_http_request_t* r);
static ngx_int_t ngx_http_add_cleanup(ngx_http_request_t* r,
                                      ngx_pool_cleanup_pt handler,
                                      void* data);
static void ngx_http_cleanup_mp4_process_context(void* context);

static ngx_int_t ngx_streaming_handler(ngx_http_request_t *r);

// the input filter and function that handle POST
static ngx_http_request_body_filter_pt ngx_http_next_request_body_filter;
static ngx_int_t
ngx_http_fmp4_request_body_filter(ngx_http_request_t *r, ngx_chain_t *cl);

#if defined(BUILDING_H264_STREAMING)
static ngx_command_t ngx_streaming_commands[] =
{
  {
    ngx_string("mp4"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_streaming,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_license_key"),
    NGX_MAIN_CONF|NGX_HTTP_MAIN_CONF|NGX_CONF_TAKE1,
    ngx_usp_license_key,
    NGX_HTTP_MAIN_CONF_OFFSET,
    0,
    NULL
  },

  ngx_null_command
};
#elif defined(BUILDING_SMOOTH_STREAMING)
static ngx_command_t ngx_streaming_commands[] =
{
  {
    ngx_string("usp_handle_ism"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_streaming,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_license_key"),
    NGX_MAIN_CONF|NGX_HTTP_MAIN_CONF|NGX_CONF_TAKE1,
    ngx_usp_license_key,
    NGX_HTTP_MAIN_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("ism_proxy_pass"),
    NGX_HTTP_LOC_CONF|NGX_CONF_TAKE1,
    ngx_ism_proxy_pass,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_prefer_static"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_prefer_static,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_iss_pass_through"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_iss_pass_through,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_dynamic_time_shift_buffer_depth"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_dynamic_time_shift_buffer_depth,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_skip_rewrite"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_skip_rewrite,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_handle_f4f"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_handle_f4f,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

    { ngx_string("usp_handle_api"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_handle_api,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("usp_disable_mmap"),
    NGX_HTTP_LOC_CONF|NGX_CONF_NOARGS,
    ngx_usp_disable_mmap,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("s3_secret_key"),
    NGX_HTTP_LOC_CONF|NGX_CONF_TAKE1,
    ngx_usp_s3_secret_key,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("s3_access_key"),
    NGX_HTTP_LOC_CONF|NGX_CONF_TAKE1,
    ngx_usp_s3_access_key,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  { ngx_string("s3_region"),
    NGX_HTTP_LOC_CONF|NGX_CONF_TAKE1,
    ngx_usp_s3_region,
    NGX_HTTP_LOC_CONF_OFFSET,
    0,
    NULL
  },

  ngx_null_command
};
#endif

static void *
ngx_http_streaming_create_loc_conf(ngx_conf_t *cf);
static char *
ngx_http_streaming_merge_loc_conf(ngx_conf_t *cf, void *parent, void *child);

static void *
ngx_http_streaming_create_main_conf(ngx_conf_t *cf);
static char *
ngx_http_streaming_init_main_conf(ngx_conf_t *cf, void *conf);

static ngx_http_module_t ngx_streaming_module_ctx =
{
  NULL,                          /* preconfiguration */
#if defined(BUILDING_H264_STREAMING)
  NULL,                          /* postconfiguration */
#elif defined(BUILDING_SMOOTH_STREAMING)
  ngx_http_streaming_init,       /* postconfiguration */
#endif

  ngx_http_streaming_create_main_conf, /* create main configuration */
  ngx_http_streaming_init_main_conf,   /* init main configuration */

  NULL,                               /* create server configuration */
  NULL,                               /* merge server configuration */

  ngx_http_streaming_create_loc_conf, /* create location configuration */
  ngx_http_streaming_merge_loc_conf   /* merge location configuration */
};

#ifdef BUILDING_H264_STREAMING
#define ngx_http_streaming_module ngx_http_h264_streaming_module
#endif
#ifdef BUILDING_SMOOTH_STREAMING
#define ngx_http_streaming_module ngx_http_smooth_streaming_module
#endif

ngx_module_t ngx_http_streaming_module =
{
  NGX_MODULE_V1,
  &ngx_streaming_module_ctx,     /* module context */
  ngx_streaming_commands,        /* module directives */
  NGX_HTTP_MODULE,               /* module type */
  NULL,                          /* init master */
  NULL,                          /* init module */
  NULL,                          /* init process */
  NULL,                          /* init thread */
  NULL,                          /* exit thread */
  NULL,                          /* exit process */
  NULL,                          /* exit master */
  NGX_MODULE_V1_PADDING
};

static void *
ngx_http_streaming_create_loc_conf(ngx_conf_t *cf)
{
  ngx_http_streaming_loc_conf_t  *conf;

  conf = ngx_pcalloc(cf->pool, sizeof(ngx_http_streaming_loc_conf_t));
  if(conf == NULL)
  {
    return NGX_CONF_ERROR;
  }

  conf->enable = NGX_CONF_UNSET;
  conf->usp_prefer_static = NGX_CONF_UNSET;
  conf->usp_iss_pass_through = NGX_CONF_UNSET;
  conf->usp_dynamic_time_shift_buffer_depth = NGX_CONF_UNSET;
  conf->usp_handle_f4f = NGX_CONF_UNSET;
  conf->usp_handle_api = NGX_CONF_UNSET;
  conf->usp_disable_mmap = NGX_CONF_UNSET;
  conf->s3_secret_key.data = NULL;
  conf->s3_secret_key.len = NGX_CONF_UNSET;
  conf->s3_access_key.data = NULL;
  conf->s3_access_key.len = NGX_CONF_UNSET;
  conf->s3_region.data = NULL;
  conf->s3_region.len = NGX_CONF_UNSET;

  return conf;
}

static char *
ngx_http_streaming_merge_loc_conf(ngx_conf_t *cf, void *parent, void *child)
{
  ngx_http_streaming_loc_conf_t *prev = parent;
  ngx_http_streaming_loc_conf_t *conf = child;

  ngx_conf_merge_value(conf->enable, prev->enable, 0);
  ngx_conf_merge_value(conf->usp_prefer_static, prev->usp_prefer_static, 0);
  ngx_conf_merge_value(conf->usp_iss_pass_through, prev->usp_iss_pass_through, 0);
  ngx_conf_merge_value(conf->usp_dynamic_time_shift_buffer_depth, prev->usp_dynamic_time_shift_buffer_depth, 0);
  ngx_conf_merge_value(conf->usp_skip_rewrite, prev->usp_skip_rewrite, 0);
  ngx_conf_merge_value(conf->usp_handle_f4f, prev->usp_handle_f4f, 0);
  ngx_conf_merge_value(conf->usp_handle_api, prev->usp_handle_api, 0);
  ngx_conf_merge_value(conf->usp_disable_mmap, prev->usp_disable_mmap, 0);

  ngx_conf_merge_str_value(conf->s3_secret_key, prev->s3_secret_key, 0);
  ngx_conf_merge_str_value(conf->s3_access_key, prev->s3_access_key, 0);
  ngx_conf_merge_str_value(conf->s3_region, prev->s3_region, 0);

  return NGX_CONF_OK;
}

static void *
ngx_http_streaming_create_main_conf(ngx_conf_t *cf)
{
  ngx_http_streaming_main_conf_t  *conf;

  conf = ngx_pcalloc(cf->pool, sizeof(ngx_http_streaming_main_conf_t));
  if(conf == NULL)
  {
    return NGX_CONF_ERROR;
  }

  conf->gctx = NGX_CONF_UNSET_PTR;
  conf->license.data = NULL;
  conf->license.len = NGX_CONF_UNSET;

  return conf;
}

static char *
ngx_http_streaming_init_main_conf(ngx_conf_t *cf, void *conf)
{
  return NGX_CONF_OK;
}

static ngx_int_t
open_path(ngx_http_request_t *r, ngx_http_core_loc_conf_t *clcf,
          ngx_str_t *path, ngx_open_file_info_t *of)
{
  ngx_uint_t level;
  ngx_int_t http_status;
  ngx_log_t *log;

  log = r->connection->log;

  ngx_memzero(of, sizeof(ngx_open_file_info_t));

#if nginx_version >= 8018
  of->read_ahead = clcf->read_ahead;
#endif
  of->directio = clcf->directio;
  of->valid = clcf->open_file_cache_valid;
  of->min_uses = clcf->open_file_cache_min_uses;
  of->errors = clcf->open_file_cache_errors;
  of->events = clcf->open_file_cache_events;

  if(ngx_open_cached_file(clcf->open_file_cache, path, of, r->pool) != NGX_OK)
  {
    switch (of->err)
    {
    case 0:
      return NGX_HTTP_INTERNAL_SERVER_ERROR;
    case NGX_ENOENT:
    case NGX_ENOTDIR:
    case NGX_ENAMETOOLONG:
      level = NGX_LOG_ERR;
      http_status = NGX_HTTP_NOT_FOUND;
      break;
    case NGX_EACCES:
      level = NGX_LOG_ERR;
      http_status = NGX_HTTP_FORBIDDEN;
      break;
    default:
      level = NGX_LOG_CRIT;
      http_status = NGX_HTTP_INTERNAL_SERVER_ERROR;
      break;
    }

    if(http_status != NGX_HTTP_NOT_FOUND || clcf->log_not_found)
    {
      ngx_log_error(level, log, of->err,
                    "%s \"%s\" failed", of->failed, path->data);
    }

    return http_status;
  }

  if(!of->is_file)
  {
    if(of->fd != NGX_INVALID_FILE && ngx_close_file(of->fd) == NGX_FILE_ERROR)
    {
      ngx_log_error(NGX_LOG_ALERT, log, ngx_errno,
                    ngx_close_file_n " \"%s\" failed", path->data);
    }
    return NGX_DECLINED;
  }

  return NGX_OK;
}

#if defined(BUILDING_SMOOTH_STREAMING)
static ngx_int_t ngx_streaming_post_rewrite_handler(ngx_http_request_t *r)
{
  ngx_http_streaming_loc_conf_t  *slcf;

  slcf = ngx_http_get_module_loc_conf(r, ngx_http_streaming_module);

  ngx_log_debug8(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                 "ngx_streaming_post_rewrite_handler: \"%V?%V\" enabled=%d"
                 " prefer_static=%d iss_pass_through=%d skip_rewrite=%d"
                 " handle_f4f=%d enable_mmap=%d",
                 &r->uri, &r->args, (int)slcf->enable,
                 (int)slcf->usp_prefer_static, (int)slcf->usp_iss_pass_through,
                 (int)slcf->usp_skip_rewrite, (int)slcf->usp_handle_f4f,
                 (int)slcf->usp_disable_mmap);

  // only rewrite when we are set up for this location
  if(!slcf->enable || slcf->usp_skip_rewrite || slcf->usp_handle_api > 0)
  {
    return NGX_DECLINED;
  }

  {
    char new_path[4096];
    char new_args[4096];
    unsigned int rewrite_ism =
      slcf->enable == NGX_CONF_UNSET ? 0 : slcf->enable;
    unsigned int rewrite_f4f =
      slcf->usp_handle_f4f == NGX_CONF_UNSET ? 0 : slcf->usp_handle_f4f;

    char const* path_first = (char const*)r->uri.data;
    char const* path_last = path_first + r->uri.len;
    char const* query_first = (char const*)r->args.data;
    char const* query_last = query_first + r->args.len;

    if(mp4_rewrite_url(path_first, path_last, query_first, query_last,
                       new_path, sizeof(new_path), new_args, sizeof(new_args),
                       rewrite_ism, rewrite_f4f))
    {
      r->uri.len = strlen(new_path);
      r->uri.data = ngx_pnalloc(r->pool, r->uri.len);
      ngx_memcpy(r->uri.data, new_path, r->uri.len);

      r->args.len = strlen(new_args);
      r->args.data = ngx_pnalloc(r->pool, r->args.len);
      ngx_memcpy(r->args.data, new_args, r->args.len);

      ngx_log_debug2(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                     "ngx_streaming_post_rewrite_handler: \"%V?%V\"",
                     &r->uri, &r->args);
    }
  }

  return NGX_DECLINED;
}
#endif

#if defined(BUILDING_SMOOTH_STREAMING)
static void ngx_http_cleanup_mp4_global_context(void* data)
{
  mp4_global_context_t* gctx = data;
  libfmp4_global_exit(gctx);
}

static ngx_int_t ngx_http_streaming_init(ngx_conf_t* cf)
{
  ngx_http_handler_pt        *h;
  ngx_http_core_main_conf_t  *cmcf;
  ngx_pool_cleanup_t         *cln;
  ngx_http_streaming_main_conf_t *smcf;
  char const* license = 0;

#if defined(BUILDING_H264_STREAMING)
  char const* src = "nginx mod_h264_streaming";
  char const* version = X_MOD_H264_STREAMING_VERSION;
#elif defined(BUILDING_SMOOTH_STREAMING)
  char const* src = "nginx mod_smooth_streaming";
  char const* version = X_MOD_SMOOTH_STREAMING_VERSION;
#endif

  mp4_global_context_t* gctx = libfmp4_global_init();

  smcf = ngx_http_conf_get_module_main_conf(cf, ngx_http_streaming_module);
  smcf->gctx = gctx;

  if(smcf->license.data)
  {
    license = (char const*)smcf->license.data;
  }

  char const* policy_check_result =
    libfmp4_load_license(gctx, src, version, license);

  if(policy_check_result != 0)
  {
    ngx_log_error(NGX_LOG_ERR, cf->log, 0, "%s", policy_check_result);
  }
  else if(license)
  {
    ngx_log_error(NGX_LOG_NOTICE, cf->log, 0, "License key found: %s", license);
  }

  // add new cleanup handler to config mem pool and register
  cln = ngx_pool_cleanup_add(cf->pool, 0);
  if(cln == NULL) {
    return NGX_ERROR;
  }
  cln->handler = ngx_http_cleanup_mp4_global_context;
  cln->data = gctx;

  cmcf = ngx_http_conf_get_module_main_conf(cf, ngx_http_core_module);

//  h = ngx_array_push(&cmcf->phases[NGX_HTTP_SERVER_REWRITE_PHASE].handlers);
//  if(h == NULL) {
//    return NGX_ERROR;
//  }
//  *h = ngx_streaming_rewrite_handler;

  h = ngx_array_push(&cmcf->phases[NGX_HTTP_REWRITE_PHASE].handlers);
  if(h == NULL) {
    return NGX_ERROR;
  }
  *h = ngx_streaming_post_rewrite_handler;

  // set the input filter to the handle function
  ngx_http_next_request_body_filter = ngx_http_top_request_body_filter;
  ngx_http_top_request_body_filter = ngx_http_fmp4_request_body_filter;

  return NGX_OK;
}
#endif

static int get_server_variable(void* context, char const* name, char const** value)
{
  ngx_http_request_t* r = (ngx_http_request_t*)context;

  if(!strcmp(name, "REQUEST_SCHEME"))
  {
#if 0 // TODO: schema_start is always NULL?
    if(r->schema_start)
    {
      *value = (char const*)r->schema_start;
      return r->schema_end - r->schema_start;
    }
#else
      *value = "http";
      return sizeof("http") - 1;
#endif
  } else
  if(!strcmp(name, "HTTP_HOST"))
  {
    if(r->headers_in.host)
    {
      *value = (char const*)r->headers_in.host->value.data;
      return r->headers_in.host->value.len;
    }
  } else
  if(!strcmp(name, "REQUEST_URI"))
  {
    *value = (char const*)r->uri.data;
    return r->uri.len;
  }

  return 0;
}

static fmp4_log_level_t nxg_log_level_to_fmp4_log_level(ngx_http_request_t* r)
{
  ngx_uint_t log_level = r->request_body_file_log_level;
  switch(log_level)
  {
  case NGX_LOG_DEBUG:
    return FMP4_LOG_DEBUG;
  case NGX_LOG_NOTICE:
    return FMP4_LOG_NOTICE_DO_NOT_USE;
  case NGX_LOG_INFO:
    return FMP4_LOG_INFO;
  case NGX_LOG_WARN:
    return FMP4_LOG_WARNING;
  case NGX_LOG_ERR:
  default:
    ; // fall through
  }
  return FMP4_LOG_ERROR;
}

static void
log_error_callback(void* context, fmp4_log_level_t level, char const* str)
{
  ngx_http_request_t* r = (ngx_http_request_t*)context;

  ngx_uint_t log_level;
  switch(level)
  {
  case FMP4_LOG_TRACE:
  case FMP4_LOG_DEBUG:
    log_level = NGX_LOG_DEBUG;
    break;
  case FMP4_LOG_INFO:
    log_level = NGX_LOG_INFO;
    break;
  case FMP4_LOG_NOTICE_DO_NOT_USE:
    log_level = NGX_LOG_NOTICE;
    break;
  case FMP4_LOG_WARNING:
    log_level = NGX_LOG_WARN;
    break;
  case FMP4_LOG_ERROR:
  default:
    log_level = NGX_LOG_ERR;
    break;
   }

  ngx_log_error(log_level, r->connection->log, 0, "fmp4: \"%s\"", str);
}

static ngx_int_t
ngx_streaming_handler_paths(ngx_http_request_t *r, mp4_process_context_t* ctx)
{
  u_char                      *last;
  size_t                      root;
  ngx_int_t                   http_status;
  ngx_str_t                   path;
  ngx_http_core_loc_conf_t    *clcf;
  ngx_http_streaming_loc_conf_t  *slcf;

  clcf = ngx_http_get_module_loc_conf(r, ngx_http_core_module);

  slcf = ngx_http_get_module_loc_conf(r, ngx_http_streaming_module);

  ctx->prefer_static_ =
    slcf->usp_prefer_static == NGX_CONF_UNSET ? 0 : slcf->usp_prefer_static;

  if(slcf->url.data)
  {
    ngx_log_debug2(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
       "ngx_streaming_handler_paths: ism_proxy_pass"
       " loc=\"%V\""
       " url=\"%V\"",
       &clcf->name, &slcf->url);
  }

  if(slcf->url.data && mp4_starts_with((char const*)r->uri.data, (char const*)clcf->name.data))
  {
    uintptr_t escape = 2 * ngx_escape_uri(NULL, r->uri.data + clcf->name.len, r->uri.len - clcf->name.len, NGX_ESCAPE_URI);
    path.len = slcf->url.len + r->uri.len - clcf->name.len + escape;
    path.data = ngx_pnalloc(r->pool, path.len + 1);
    last = ngx_copy(path.data, slcf->url.data, slcf->url.len);
    ngx_escape_uri(last, r->uri.data + clcf->name.len, r->uri.len - clcf->name.len, NGX_ESCAPE_URI);
    last += r->uri.len - clcf->name.len + escape;
    *last++ = '\0';
  }
  else
  {
    last = ngx_http_map_uri_to_path(r, &path, &root, 0);

    if(last == NULL)
    {
      return NGX_HTTP_INTERNAL_SERVER_ERROR;
    }

    path.len = last - path.data;
    path.data[path.len] = '\0';

    ngx_log_debug1(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                   "ngx_streaming_handler_path: ngx_http_map_uri_to_path path=\"%V\"",
                   &path);

    ngx_open_file_info_t of;
    http_status = open_path(r, clcf, &path, &of);

#if 0 // don't rely on (stale) file metadata
    if(http_status == NGX_OK)
    {
      ctx->filesize_ = of.size;
      r->headers_out.last_modified_time = of.mtime;
    }
#endif

    // a file-not-found may occur when ingesting a live stream that has no server manifest file.
    if(http_status != NGX_OK && http_status != NGX_HTTP_NOT_FOUND)
    {
      return http_status;
    }
  }

  ctx->filename_ = ngx_pcalloc(r->pool, path.len+1);
  ngx_memcpy((void*)ctx->filename_, path.data, path.len);

  ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
               "ngx_streaming_handler_paths: exit");

  return NGX_OK;
}

static ngx_int_t ngx_streaming_handler(ngx_http_request_t *r)
{
  ngx_http_streaming_loc_conf_t  *slcf;
  ngx_http_streaming_main_conf_t  *smcf;

  if(r->content_handler != ngx_streaming_handler)
  {
    return NGX_DECLINED;
  }

  ngx_log_debug2(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                 "ngx_streaming_handler: \"%V?%V\"", &r->uri, &r->args);

  ngx_uint_t allowed = r->method & (NGX_HTTP_GET|NGX_HTTP_HEAD|NGX_HTTP_POST|NGX_HTTP_PUT);

  slcf = ngx_http_get_module_loc_conf(r, ngx_http_streaming_module);

  if(slcf->usp_handle_api > 0)
  {
    allowed = r->method & (NGX_HTTP_GET|NGX_HTTP_POST|NGX_HTTP_PUT|NGX_HTTP_DELETE);
  }

  if(!allowed)
  {
    return NGX_HTTP_NOT_ALLOWED;
  }

  if(r->uri.data[r->uri.len - 1] == '/')
  {
    return NGX_DECLINED;
  }

#if defined(nginx_version) && \
    ((nginx_version < 7066) || \
     ((nginx_version > 8000) && (nginx_version < 8038)))
  if(r->zero_in_uri)
  {
    return NGX_DECLINED;
  }
#endif

  // add X-USP header
  ngx_table_elt_t* h = ngx_list_push(&r->headers_out.headers);
  if(h == NULL)
  {
    return NGX_HTTP_INTERNAL_SERVER_ERROR;
  }
  h->hash = 1;
  h->key.len = sizeof("X-USP") - 1;
  h->key.data = (u_char*)"X-USP";
  h->value.len = strlen(fmp4_version_string());
  h->value.data = (u_char*)fmp4_version_string();

  ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                "ngx_streaming_handler: mp4-process-context-init");

  smcf = ngx_http_get_module_main_conf(r, ngx_http_streaming_module);

  mp4_process_context_t* context = ngx_palloc(r->pool, sizeof(mp4_process_context_t));
  mp4_process_context_init(context, smcf->gctx);
  ngx_http_add_cleanup(r, ngx_http_cleanup_mp4_process_context, context);

  context->get_server_variable_callback_ = get_server_variable;
  context->get_server_variable_context_ = r;
  context->log_error_callback_ = log_error_callback;
  context->log_error_context_ = r;
  context->verbose_ = nxg_log_level_to_fmp4_log_level(r);
  context->method_ = ngx_get_method(r);

  context->prefer_static_ = slcf->usp_prefer_static == NGX_CONF_UNSET ? 0
                          : slcf->usp_prefer_static;
  context->iss_pass_through_ = slcf->usp_iss_pass_through == NGX_CONF_UNSET ? 0
                             : slcf->usp_iss_pass_through;
  context->dynamic_time_shift_buffer_depth_ =
    slcf->usp_dynamic_time_shift_buffer_depth == NGX_CONF_UNSET ? 0 :
    slcf->usp_dynamic_time_shift_buffer_depth;

  if(slcf->usp_disable_mmap)
  {
    context->create_handler_io_ = create_handler_io_no_mmap;
  }

  if(slcf->s3_secret_key.data && slcf->s3_access_key.data)
  {
    context->s3_secret_key_ = (char const*)slcf->s3_secret_key.data;
    context->s3_access_key_ = (char const*)slcf->s3_access_key.data;
    if(slcf->s3_region.data)
    {
      context->s3_region_ = (char const*)slcf->s3_region.data;
    }
  }
  ngx_int_t http_status = ngx_streaming_handler_paths(r, context);
  if(NGX_OK != http_status)
  {
    //mp4_process_context_exit(context);
    return http_status;
  }

  mp4_split_options_t* options = context->options_;
  if(r->args.len)
  {
    if(!mp4_split_options_set(options,
        (const char*)r->args.data, r->args.len))
    {
      //mp4_process_context_exit(context);
      return NGX_HTTP_BAD_REQUEST;
    }
  }

  if(slcf->usp_handle_api > 0)
  {
    context->is_rest_api_ = 1;
  }

  if(r->method & (NGX_HTTP_GET|NGX_HTTP_HEAD))
  {
    http_status = ngx_streaming_handler_get(r, context);
  }
#if defined(BUILDING_SMOOTH_STREAMING)
  else if(r->method & (NGX_HTTP_POST|NGX_HTTP_PUT))
  {
    http_status = ngx_streaming_handler_post(r, context);
  }

  if(r->method & NGX_HTTP_DELETE && context->is_rest_api_)
  {
    http_status = mp4_process(context);
    r->headers_out.content_length_n = 0;
    r->headers_out.status = http_status;
    r->header_only = 1;
    http_status = ngx_http_send_header(r);
  }
#endif

  //mp4_process_context_exit(context);

  // Note: do not do any logging here to r->connection->log. The connection may
  // already have been finalized.
//ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
//              "ngx_streaming_handler: mp4-process-context-exit");

  return http_status;
}

#if defined(BUILDING_SMOOTH_STREAMING)
static ngx_int_t
write_to_pubpoint(post_handler_t* post_handler, unsigned char* first,
                  unsigned char* last, ngx_http_request_t *r)
{
  char result_text[256] = { 0 };

  int result = post_handler_insert(post_handler, first, last,
    result_text, sizeof(result_text) / sizeof(result_text[0]));

  if(result != NGX_HTTP_OK)
  {
    ngx_log_error(NGX_LOG_ERR, r->connection->log, 0,
                  "pubpoint_push_input_stream returned: %i", result);
    if(result_text[0])
    {
      ngx_log_error(NGX_LOG_ERR, r->connection->log, 0,
                    "pubpoint_push_input_stream message: %s", result_text);
    }

    // abort the connection in case we are receiving the POST data as
    // chunked. Unfortunately we lose the original http_status
    // code and Apache always returns a 500 Internal Server Error.
    if(first != last)
    {
      result = NGX_HTTP_INTERNAL_SERVER_ERROR;
    }
  }

  return result;
}

static void ngx_http_fmp4_input_post_read(ngx_http_request_t *r)
{
  ngx_http_streaming_loc_conf_t* slcf =
    ngx_http_get_module_loc_conf(r, ngx_http_streaming_module);

  if(slcf->usp_handle_api <= 0)
  {
    ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                  "ngx_http_fmp4_input_post_read: post_handler_exit");
  }

  r->read_event_handler = ngx_http_request_empty_handler;

  ngx_int_t http_status = NGX_HTTP_OK;

  // get the handler from the request context
  ngx_http_streaming_ctx_t* ctx =
    ngx_http_get_module_ctx(r, ngx_http_streaming_module);

  http_status = write_to_pubpoint(ctx->handler, NULL, NULL, r);
  post_handler_exit(ctx->handler);

  r->headers_out.content_length_n = 0;
  r->headers_out.status = http_status;
  r->header_only = 1;
  r->keepalive = 0;

  ngx_http_finalize_request(r, ngx_http_send_header(r));
}

static ngx_int_t
ngx_http_fmp4_request_body_filter(ngx_http_request_t* r,
                                  ngx_chain_t* in)
{
  ngx_chain_t* cl = NULL;
  ngx_http_streaming_loc_conf_t* slcf =
    ngx_http_get_module_loc_conf(r, ngx_http_streaming_module);

  if(slcf->enable)
  {
    // get the handler from the request context
    ngx_http_streaming_ctx_t* ctx =
      ngx_http_get_module_ctx(r, ngx_http_streaming_module);

    ngx_log_debug1(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                   "ngx_http_fmp4_request_body_filter post_handler=%p",
                   ctx->handler);

    for(cl = in; cl != NULL; cl = cl->next)
    {
      ngx_int_t http_status = NGX_HTTP_OK;

      if(ctx->handler)
      {
        size_t len = (size_t) (cl->buf->last - cl->buf->pos);

        ngx_log_debug1(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                       "write_to_pubpoint size=%i", len);
        if(len)
        {
          http_status =
            write_to_pubpoint(ctx->handler, cl->buf->pos, cl->buf->last, r);

          // discard the body bytes so they are not written to tmp file later
          cl->buf->pos = cl->buf->last;
        }
      }

      if(http_status != NGX_HTTP_OK)
      {
        r->headers_out.content_length_n = 0;
        r->headers_out.status = http_status;
        r->header_only = 1;
        r->keepalive = 0;
        ngx_http_finalize_request(r, ngx_http_send_header(r));

        return NGX_ERROR;
      }
    }
  }

  return ngx_http_next_request_body_filter(r, in);
}

static ngx_int_t
ngx_streaming_handler_post(ngx_http_request_t *r, mp4_process_context_t* ctx)
{
  ngx_int_t http_status;

  if(!mp4_ends_with(ctx->filename_, ".isml"))
  {
    return NGX_HTTP_FORBIDDEN;
  }

  mp4_split_options_t const* options = ctx->options_;
  char const* file = mp4_split_options_get_file(options);

  // Connect
  if(!file[0] && !ctx->is_rest_api_)
  {
    r->headers_out.content_length_n = 0;
    r->headers_out.status = NGX_HTTP_OK;
    r->header_only = 1;
    http_status = ngx_http_send_header(r);
    return http_status;
  }

  // Pusing to .isml/Streams(...) disable api_context
  if (file[0] && ctx->is_rest_api_)
  {
    ctx->is_rest_api_ = 0;
  }

  if(!strcmp(file, "purge"))
  {
    fmp4_result result = mp4_process(ctx);

    if(result != NGX_HTTP_OK)
    {
      ngx_log_error(NGX_LOG_ERR, r->connection->log, 0,
                    "mp4_process(%s/%s) returned: %s %s",
                    ctx->filename_,
                    file,
                    fmp4_result_to_string(ctx->result_),
                    ctx->result_text_);

      return result;
    }

    r->headers_out.content_length_n = 0;
    r->headers_out.status = result;
    r->header_only = 1;
    http_status = ngx_http_send_header(r);

    return http_status;
  }

  ngx_log_debug1(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                "ngx_streaming_handler_post: manifest=%s",
                (char const*)ctx->filename_);

  post_handler_t* post_handler = create_post_handler(ctx);

  if(post_handler == NULL)
  {
    ngx_log_debug(NGX_LOG_ERR, r->connection->log, 0,
                  "usp_post_handler returned: %s", ctx->result_text_);

    return NGX_HTTP_INTERNAL_SERVER_ERROR;
  }

  ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                "ngx_streaming_handler_post: create_post_handler");

  // attach the handler to the request context to be used in the input filter
  ngx_http_streaming_ctx_t* rctx =
    ngx_http_get_module_ctx(r, ngx_http_streaming_module);
  if(rctx == NULL)
  {
    rctx = ngx_pcalloc(r->pool, sizeof(ngx_http_streaming_ctx_t));
    if(rctx == NULL)
    {
      ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
              "ngx_streaming_handler_post: could not create req handler ctx");
       return NGX_ERROR;
    }
    rctx->handler = post_handler;
    ngx_http_set_ctx(r, rctx, ngx_http_streaming_module);
  }

  // make sure this flag is up, otherwise we may be missing the last part of the body
  r->request_body_in_file_only = 1;

  http_status = ngx_http_read_client_request_body(r, ngx_http_fmp4_input_post_read);

  ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, http_status,
                "ngx_streaming_handler_post: exit");

  if(http_status >= NGX_HTTP_SPECIAL_RESPONSE)
  {
    return http_status;
  }

  return NGX_DONE;
}
#endif // BUILDING_SMOOTH_STREAMING

static ngx_int_t
ngx_streaming_handler_get(ngx_http_request_t *r,
                          mp4_process_context_t* context)
{
  ngx_int_t                   http_status;
  ngx_log_t                   *log;
  ngx_http_core_loc_conf_t    *clcf;

  http_status = ngx_http_discard_request_body(r);

  if(http_status != NGX_OK)
  {
    return http_status;
  }

  clcf = ngx_http_get_module_loc_conf(r, ngx_http_core_module);

  mp4_split_options_t* options = context->options_;
  char const* file = mp4_split_options_get_file(options);

  ngx_log_debug(NGX_LOG_DEBUG_HTTP, r->connection->log, 0,
                "ngx_streaming_handler_get: %s", (char const*)context->filename_);

  // If we are requesting the server manifest file, then it may an encoding tool
  // that is using the GET request to get the authorization setup for the POST.
  if(mp4_ends_with((char const*)context->filename_, ".isml"))
  {
    if(mp4_starts_with(file, "Streams("))
    {
      //mp4_process_context_exit(context);
      r->headers_out.content_length_n = 0;
      r->headers_out.status = NGX_HTTP_OK;
      r->header_only = 1;
      http_status = ngx_http_send_header(r);
      return http_status;
    }

//    // we are either serving dynamically generated files, or fragments
//    if(!file[0])
//    {
//      mp4_process_context_exit(&context);
//      return NGX_HTTP_FORBIDDEN;
//    }
  }

  log = r->connection->log;

  // ??
  r->root_tested = !r->error_page;

  {
    r->headers_out.status = http_status = mp4_process(context);

    headers_t* headers = context->headers_;
    buckets_t* buckets = context->buckets_;

    if(headers->x_usp_header1[0])
    {
      ngx_table_elt_t* h = ngx_list_push(&r->headers_out.headers);
      if(h == NULL)
      {
        return NGX_HTTP_INTERNAL_SERVER_ERROR;
      }
      h->hash = 1;

      h->key.len = sizeof("X-USP-Info1") - 1;
      h->key.data = (u_char*)"X-USP-Info1";
      h->value.len = strlen(headers->x_usp_header1);
      h->value.data = (u_char*)headers->x_usp_header1;
    }

    if(headers->x_usp_header2[0])
    {
      ngx_table_elt_t* h = ngx_list_push(&r->headers_out.headers);
      if(h == NULL)
      {
        return NGX_HTTP_INTERNAL_SERVER_ERROR;
      }
      h->hash = 1;

      h->key.len = sizeof("X-USP-Info2") - 1;
      h->key.data = (u_char*)"X-USP-Info2";
      h->value.len = strlen(headers->x_usp_header2);
      h->value.data = (u_char*)headers->x_usp_header2;
    }

    // Add Last-Modified
    if(headers->updated_at_)
    {
      r->headers_out.last_modified_time = headers->updated_at_ / 1000000;
    }

    // compute expiry with seconds accuracy for interoperable results with nginx
    typedef uint64_t seconds_t;

    // Add X-USP-Last-Modified
    if(headers->updated_at_experimental_)
    {
      seconds_t updated_time = headers->updated_at_experimental_ / 1000000;
      ngx_table_elt_t* h = ngx_list_push(&r->headers_out.headers);
      if(h == NULL)
      {
        return NGX_HTTP_INTERNAL_SERVER_ERROR;
      }
      h->hash = 1;
      h->key.len = sizeof("X-USP-Last-Modified") - 1;
      h->key.data = (u_char*)"X-USP-Last-Modified";
      h->value.len = sizeof("Mon, 28 Sep 1970 06:00:00 GMT") - 1;
      h->value.data =
        ngx_pnalloc(r->pool, sizeof("Mon, 28 Sep 1970 06:00:00 GMT"));
      if(h->value.data == NULL)
      {
        return NGX_ERROR;
      }
      ngx_http_time(h->value.data, updated_time);
    }

     // Add Cache-Control and Expires (see also ngx_http_headers_filter_module.c)
    {
      ngx_str_t cache_control = ngx_null_string;
      ngx_str_t expires = ngx_null_string;

      if(headers->cache_control_)
      {
        cache_control.len = strlen(headers->cache_control_);
        cache_control.data = (u_char*)headers->cache_control_;
      }
      else if(headers->expires_at_)
      {
        seconds_t expires_time = headers->expires_at_ / 1000000;
        seconds_t request_time = r->start_sec;

        // use expiry from backend only if it's at least a second ahead of request
        // otherwise, we skip insertion of Expires/Cache-Control and leave this
        // up to the customer (see: add_header, expires)
        if(expires_time < request_time + 1)
        {
          expires_time = request_time + 1;
        }

        seconds_t max_age = expires_time - request_time;

        cache_control.data =
          ngx_pnalloc(r->pool, sizeof("max-age=") + NGX_TIME_T_LEN + 1);
        if(cache_control.data == NULL)
        {
          return NGX_ERROR;
        }
        cache_control.len =
          ngx_sprintf(cache_control.data, "max-age=%T", max_age) - cache_control.data;

        expires.len = sizeof("Mon, 28 Sep 1970 06:00:00 GMT") - 1;
        expires.data =
          ngx_pnalloc(r->pool, sizeof("Mon, 28 Sep 1970 06:00:00 GMT"));
        if(expires.data == NULL)
        {
          return NGX_ERROR;
        }
        ngx_http_time(expires.data, expires_time);
      }

      if(cache_control.len)
      {
        ngx_table_elt_t** ccp = r->headers_out.cache_control.elts;
        ngx_table_elt_t* cc;

        if(ccp == NULL)
        {
          if(ngx_array_init(&r->headers_out.cache_control, r->pool,
                            1, sizeof(ngx_table_elt_t *)) != NGX_OK)
          {
            return NGX_ERROR;
          }

          ccp = ngx_array_push(&r->headers_out.cache_control);
          if(ccp == NULL)
          {
            return NGX_ERROR;
          }

          cc = ngx_list_push(&r->headers_out.headers);
          if(cc == NULL)
          {
            return NGX_ERROR;
          }

          cc->hash = 1;
          ngx_str_set(&cc->key, "Cache-Control");
          *ccp = cc;
        }
        else
        {
          ngx_uint_t i;
          for(i = 1; i < r->headers_out.cache_control.nelts; ++i)
          {
            ccp[i]->hash = 0;
          }
          cc = ccp[0];
        }

        cc->value = cache_control;
      }

      if(expires.len)
      {
        if(r->headers_out.expires == NULL)
        {
          r->headers_out.expires = ngx_list_push(&r->headers_out.headers);
          if(r->headers_out.expires == NULL)
          {
            return NGX_ERROR;
          }
          r->headers_out.expires->hash = 1;
          ngx_str_set(&r->headers_out.expires->key, "Expires");
        }
        r->headers_out.expires->value = expires;
      }
    }

    // Add Sunset header
    if(headers->sunset_at_)
    {
      seconds_t sunset_time = headers->sunset_at_ / 1000000;

      ngx_table_elt_t* h = ngx_list_push(&r->headers_out.headers);
      if(h == NULL)
      {
        return NGX_HTTP_INTERNAL_SERVER_ERROR;
      }
      h->hash = 1;
      h->key.len = sizeof("Sunset") - 1;
      h->key.data = (u_char*)"Sunset";
      h->value.len = sizeof("Mon, 28 Sep 1970 06:00:00 GMT") - 1;
      h->value.data = ngx_pnalloc(r->pool,
                                  sizeof("Mon, 28 Sep 1970 06:00:00 GMT"));
      if(h->value.data == NULL)
      {
        return NGX_ERROR;
      }
      ngx_http_time(h->value.data, sunset_time);
    }

    // Add the link header for start/next
    if(headers->link_[0])
    {
      ngx_table_elt_t* h = ngx_list_push(&r->headers_out.headers);
      if(h == NULL)
      {
        //mp4_process_context_exit(&context);
        return NGX_HTTP_INTERNAL_SERVER_ERROR;
      }
      h->hash = 1;

      h->key.len = sizeof("Link") - 1;
      h->key.data = (u_char*)"Link";
      h->value.len = strlen(headers->link_);
      h->value.data = (u_char*)headers->link_;
    }

    // Set the content length
    {
      uint64_t content_length = buckets_size(buckets);

      if(content_length == UINT64_MAX)
      {
        return 501; /* Not Implemented */
      }

      //r->headers_out.status = NGX_HTTP_OK;
      r->headers_out.content_length_n = content_length;

      // Nginx doesnt understand content_length_n=0 and headers_only=0, stalls
      if(content_length == 0)
      {
        r->header_only = 1;
      }
    }

    // Add ETag header
    if(headers->etag_[0])
    {
      ngx_table_elt_t* etag = ngx_list_push(&r->headers_out.headers);
      if(etag == NULL)
      {
        return NGX_ERROR;
      }

      etag->hash = 1;
      ngx_str_set(&etag->key, "ETag");

      etag->value.len = strlen(headers->etag_);
      etag->value.data = (u_char*)headers->etag_;

      r->headers_out.etag = etag;
    }
    else
    {
      if(ngx_http_set_etag(r) != NGX_OK)
      {
        return NGX_HTTP_INTERNAL_SERVER_ERROR;
      }
    }

    if(r->headers_out.status != 200)
    {
      ngx_log_error(NGX_LOG_ERR, log, 0,
                    "mp4_process(%s/%s) returned: %s %s",
                    context->filename_,
                    file,
                    fmp4_result_to_string(context->result_),
                    context->result_text_);
    }
    else
    {
      // Set the content type
      r->headers_out.content_type.len = strlen(headers->content_type_);
      r->headers_out.content_type.data = (u_char*)headers->content_type_;
    }

    // Nginx properly handles range requests, even when we're sending chunks
    // from both memory (the metadata) and part file (the movie data).
    r->allow_ranges = 1;

    int headers_out_status = r->headers_out.status;

    http_status = ngx_http_send_header(r);

    if(http_status == NGX_ERROR || http_status > NGX_OK || r->header_only || headers_out_status != 200)
    {
      return http_status;
    }

    {
      ngx_chain_t* out = 0;
      ngx_chain_t** chain = &out;
      bucket_t* head = buckets->bucket_;
      bucket_t* bucket = bucket_next(head);

      log->action = "sending mp4 to client";

      if(bucket != head)
      {
        ngx_str_t path;
        ngx_open_file_info_t of;
        char const* filename_prev = 0;

        for(; bucket != head; bucket = bucket_next(bucket))
        {
          ngx_buf_t* b = ngx_pcalloc(r->pool, sizeof(ngx_buf_t));
          *chain = ngx_pcalloc(r->pool, sizeof(ngx_chain_t));
          if(b == NULL || *chain == NULL)
          {
            //mp4_process_context_exit(&context);
            return NGX_HTTP_INTERNAL_SERVER_ERROR;
          }

          if(is_bucket_type_file(bucket))
          {
            char const* fname;
            uint64_t offset;
            uint64_t size = 0;
            uint32_t max_size = 16 * 1024 * 1024;
            fmp4_result result = bucket_file_read(bucket, &fname, &offset, &size, max_size);
            if(result != FMP4_OK)
            {
              ngx_log_error(NGX_LOG_ERR, log, 0, "bucket_file_read failed: %s",
                            fmp4_result_to_string(result));
            }

            b->file = ngx_pcalloc(r->pool, sizeof(ngx_file_t));
            if(b->file == NULL)
            {
              //mp4_process_context_exit(&context);
              return NGX_HTTP_INTERNAL_SERVER_ERROR;
            }

            // poor man's caching
            if(fname != filename_prev)
            {
              filename_prev = fname;
              {
                char* dst;

                path.len = strlen(fname);
                path.data = ngx_pnalloc(r->pool, path.len + 1);
                ngx_memcpy(path.data, fname, path.len + 1);

                // convert from a escaped file:// URL to a local path
                dst = (char*)path.data;
                path.len = file_url_to_path(dst) - dst;
              }

              http_status = open_path(r, clcf, &path, &of);

              if(http_status != NGX_OK)
              {
                //mp4_process_context_exit(&context);
                return http_status;
              }
            }

            b->file->fd = of.fd;
            b->file->name = path;
            b->file->log = log;
            b->file_pos = offset;
            b->file_last = b->file_pos + size;
            b->in_file = b->file_last ? 1: 0;
          }
          else
          {
            uint8_t const* src = 0;
            size_t size = 0;
            fmp4_result result = bucket_read(bucket, &src, &size);
            if(result != FMP4_OK)
            {
              ngx_log_error(NGX_LOG_ERR, log, 0, "bucket_read failed: %s",
                            fmp4_result_to_string(result));
            }

            b->pos = ngx_pcalloc(r->pool, size);
            if(b->pos == NULL)
            {
              //mp4_process_context_exit(&context);
              return NGX_HTTP_INTERNAL_SERVER_ERROR;
            }
            b->last = b->pos + size;
            b->memory = 1;
            memcpy(b->pos, src, size);
          }

          bucket_t* next = bucket_next(bucket);
          b->last_buf = next == head ? 1 : 0;
          b->last_in_chain = next == head ? 1 : 0;

          (*chain)->buf = b;
          (*chain)->next = NULL;
          chain = &(*chain)->next;
        }
        //mp4_process_context_exit(&context);
      }

      return ngx_http_output_filter(r, out);
    }
  }
}

static char* ngx_streaming(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_core_loc_conf_t* clcf;
  ngx_http_streaming_loc_conf_t* slcf = conf;

  clcf = ngx_http_conf_get_module_loc_conf(cf, ngx_http_core_module);

  clcf->handler = ngx_streaming_handler;

  slcf->enable = 1;

  return NGX_CONF_OK;
}

#if defined(BUILDING_SMOOTH_STREAMING)
static char*
ngx_ism_proxy_pass(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  ngx_str_t* value = cf->args->elts;
  ngx_str_t* url = &value[1];

  slcf->url = *url;
  return NGX_CONF_OK;
}

static char*
ngx_usp_prefer_static(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  slcf->usp_prefer_static = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_iss_pass_through(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  slcf->usp_iss_pass_through = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_dynamic_time_shift_buffer_depth(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  slcf->usp_dynamic_time_shift_buffer_depth = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_skip_rewrite(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  slcf->usp_skip_rewrite = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_handle_f4f(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  slcf->usp_handle_f4f = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_handle_api(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_core_loc_conf_t* clcf;
  ngx_http_streaming_loc_conf_t* slcf = conf;

  clcf = ngx_http_conf_get_module_loc_conf(cf, ngx_http_core_module);

  clcf->handler = ngx_streaming_handler;

  slcf->enable = 1;

  slcf->usp_handle_api = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_disable_mmap(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  slcf->usp_disable_mmap = 1;

  return NGX_CONF_OK;
}

static char* ngx_usp_s3_secret_key(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  ngx_str_t* value = cf->args->elts;
  ngx_str_t* key = &value[1];
  slcf->s3_secret_key = *key;

  return NGX_CONF_OK;
}

static char* ngx_usp_s3_access_key(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  ngx_str_t* value = cf->args->elts;
  ngx_str_t* key = &value[1];
  slcf->s3_access_key = *key;

  return NGX_CONF_OK;
}

static char* ngx_usp_s3_region(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_loc_conf_t* slcf = conf;

  ngx_str_t* value = cf->args->elts;
  ngx_str_t* key = &value[1];
  slcf->s3_region = *key;

  return NGX_CONF_OK;
}

#endif

static char*
ngx_usp_license_key(ngx_conf_t* cf, ngx_command_t* cmd, void* conf)
{
  ngx_http_streaming_main_conf_t* smcf;
  smcf = ngx_http_conf_get_module_main_conf(cf, ngx_http_streaming_module);

  ngx_str_t* value = cf->args->elts;
  ngx_str_t* license = &value[1];

  smcf->license = *license;

  return NGX_CONF_OK;
}

static fmp4_http_method_t ngx_get_method(ngx_http_request_t* r)
{
  switch (r->method)
  {
  case NGX_HTTP_GET:
    return FMP4_HTTP_GET;
  case NGX_HTTP_PUT:
    return FMP4_HTTP_PUT;
  case NGX_HTTP_POST:
    return FMP4_HTTP_POST;
  case NGX_HTTP_DELETE:
    return FMP4_HTTP_DELETE;
  default:
    return r->method;
  }
}

static ngx_int_t ngx_http_add_cleanup(ngx_http_request_t* r,
                                      ngx_pool_cleanup_pt handler,
                                      void* data)
{
  ngx_pool_cleanup_t* cln = ngx_pool_cleanup_add(r->pool, 0);

  if(cln == NULL)
  {
    return NGX_ERROR;
  }

  cln->handler = handler;
  cln->data = data;

  return NGX_OK;
}

static void ngx_http_cleanup_mp4_process_context(void* context)
{
  mp4_process_context_exit((mp4_process_context_t*)context);
}

// End Of File

